/* WOMinator Bass Pedals Sketch */
/* (c) 2012 Lee O'Donnell */
/* http://www.bassmaker.co.uk */
/* lee@bassmaker.co.uk */

/* Include the SPI/IIC Library */
#include <Wire.h>
#include <LiquidCrystal_I2C.h>
#include <NewSoftSerial.h>    // Library for software uart for MIDI

#define LeeDebug    0         // 1 = print debug messages to serial port, 0 = don't
#define DebugMidi   0         // If 1 will echo all midi commands to the serial port for debugging
#define midiBridge  1         // If 1 will send real midi data via the serial port. Make sure both debug settings are off! It also doesn't support program changes
#define midichan 1  // Midi channel to use

#define midiNoteStart 36  // C1 key is this note value, the rest go up

#define octDownLEDPin 9   // Pin for octave down LED
#define octUpLEDPin 10   // Pin for octave up LED
#define modeLEDPin 11   // Pin for  mode LED

int octDownLED=0;  // PWM value for octave down LED
int octDownRate=0; // PWM rate for octave down LED
int octUpLED=0;  // PWM value for octave down LED
int octUpRate=0;     // PWM rate for octave up LED

int modeLED=0;  // PWM value for mode LED
int modeRate=0; // PWM rate for mode LED

int playMode=0;  // Mode=0 is normal, Mode=1 is mono/sustain

char input_port=0;  // port char

int globTranspose=0;        // Global variable for transposing purposes
int progNumber=1;  // Keep track of what midi program number we're on, default to patch 1

int lastGlobTranspose=0;  // Used for LCD update
int lastProgNumber=1;    // Used for LCD update
int lastPlayMode=0;       // Used for LCD update

int currentNoteVal=0;    // If mono mode, this is the note currently playing

unsigned char lastKeyActive[13];  // Array to hold which keys were active on last loop 0..12 = 13 pedals
unsigned char keyActive[13];  // Array to hold which keys were active on this loop
unsigned int keyTimer[13];  // Array to hold timers for measuring key velocity

unsigned char lastButtonActive[6];  // Array to hold which function buttons were active
unsigned char buttonActive[6];  // Array to hold which function buttons are active

/* Initialise the LiquidCrystal library. The default address is 0x27 and this is a 16x2 line display */
LiquidCrystal_I2C lcd(0x27,16,2);

NewSoftSerial mySerial(20, 8);    // Create instance of soft serial port for MIDI 8=D8 out, 20=Analog 6 in (not used)

void setup() 
{
  /* Initialise the LCD */
  lcd.init();
  lcd.init();
  /* Make sure the backlight is turned on */
  lcd.backlight();
  lcd.setCursor(0,0);
  lcd.print(" *WOMinator* (c)"); 
  lcd.setCursor(0,1);
  lcd.print("2012 L O'Donnell"); 

 DDRB=B11111111; // All outputs for matrix and LEDs (bits 6 and 7 not used)
 DDRC=B11111111;	// All outputs
 DDRD=B00000010; // All inputs except  USB TX

 for (int i=2; i<8; i++)  // Setup the row pins 2..7 as INPUTs
 {
  digitalWrite(i,HIGH);  // Turn on the pull-up resistors on the inputs
 }

 for (int i=12; i<18; i++)  // Turn pins 12-17 high (inactive)
 {
  digitalWrite(i,HIGH);  // Turn on
 }

for (int i=0;i<13;i++)  // Ensure arrays are cleared
 {
  keyActive[i]=0;
  lastKeyActive[i]=0;
  keyTimer[i]=0;
 }

 mySerial.begin(31250);  // Initialise Midi Out port
 Serial.begin(38400);    // Debugging output on normal UART back through USB 
  delay(2000);    // Time to read startup message
  lcd.setCursor(0,0);
  lcd.print("PROG OCT CH MODE"); 
  lcd.setCursor(0,1);
  lcd.print("   1   0 01 POLY"); 
}

/* Main program loop */
void loop() 
{
   scanKeys();  // Scan the key inputs
   outputMIDI(); // DWISOTT
   scanButtons(); // Scan the function buttons and process actions
   doPWM();       // Control LED outputs
   updateLCD();
   
delay(1); // stop it going too fast around the loop
}

void scanKeys()
{
  for (int i=0; i<5; i++)  // Turn pins 12-17 high (inactive) -- should be 18 once function keys in place
 {
  digitalWrite(i+12,LOW);  // Turn low
  input_port=PIND;
 
   if( ( ((input_port & B00000100)>>2) ==1) && ((input_port & B00001000)>>3) ==0)  { keyActive[(i*3)]=1; } else { keyActive[(i*3)]=0; }    // NC contact open, NO contact closed = key on, array loc 0,6,12

   if( ( ((input_port & B00000100)>>2) ==1) && (((input_port & B00001000)>>3) ==1) && (keyActive[(i*3)]==0) && (keyTimer[(i*3)]!=0))  { keyTimer[(i*3)]--; if (keyTimer[(i*3)]<0) {keyTimer[(i*3)]=0;} }    // previously off key in transition
   if( ( ((input_port & B00000100)>>2) ==1) && (((input_port & B00001000)>>3) ==1) && (keyActive[(i*3)]==0) && (keyTimer[(i*3)]==0))  { keyTimer[(i*3)]=127; }    // previously off key in transition

 if(i<4) {  // Row 4,5 is special case as only 1 key
   if( ( ((input_port & B00010000)>>4) ==1) && ((input_port & B00100000)>>5) ==0)  { keyActive[(i*3)+1]=1; } else { keyActive[(i*3)+1]=0; } // NC contact open, NO contact closed = key on
   if( ( ((input_port & B01000000)>>6) ==1) && ((input_port & B10000000)>>7) ==0)  { keyActive[(i*3)+2]=1; } else { keyActive[(i*3)+2]=0; } // NC contact open, NO contact closed = key on

   if( ( ((input_port & B00010000)>>4) ==1) && (((input_port & B00100000)>>5) ==1) && (keyActive[(i*3)+1]==0) && (keyTimer[(i*3)+1]!=0))  { keyTimer[(i*3)+1]--; if (keyTimer[(i*3)]<0) {keyTimer[(i*3)]=0;} }    // previously off key in transition
   if( ( ((input_port & B00010000)>>4) ==1) && (((input_port & B00100000)>>5) ==1) && (keyActive[(i*3)+1]==0) && (keyTimer[(i*3)+1]==0))  { keyTimer[(i*3)+1]=127; }    // previously off key in transition

   if( ( ((input_port & B01000000)>>6) ==1) && (((input_port & B10000000)>>7) ==1) && (keyActive[(i*3)+2]==0) && (keyTimer[(i*3)+2]!=0))  { keyTimer[(i*3)+2]--; if (keyTimer[(i*3)+2]<0) {keyTimer[(i*3)+2]=0;} }    // previously off key in transition
   if( ( ((input_port & B01000000)>>6) ==1) && (((input_port & B10000000)>>7) ==1) && (keyActive[(i*3)+2]==0) && (keyTimer[(i*3)+2]==0))  { keyTimer[(i*3)+2]=127; }    // previously off key in transition

   }   

// for velocity sensitivity, we need to watch for NC and NO bit both being 1 (pedal in transition) when it was previously keyActive=0 and start counting. Stop when NC=1 and NO=0 (contact made)

  digitalWrite(i+12,HIGH);  // Turn high = inactive
 }
}

void outputMIDI()
{
  int velValue;
  int lastNoteVal=0;
  if (playMode==0)  // Polyphonic = process all as normal
   {
     for (int i=0; i<13; i++)  // Go through the 13 notes (0..12) on the keyboard
     {
           if(keyActive[i]==1 && lastKeyActive[i]==0) // Note on since last time
          { 
            velValue=15+((keyTimer[i]-64)<<1);
            if(velValue<0) { velValue=0;} else if(velValue>127) { velValue=127; }
            turnNoteOn(i+midiNoteStart,velValue); 
            lastKeyActive[i]=1; 
            //lcd.setCursor(0,0);
            //lcd.print(keyTimer[i]); 
            //lcd.print("  "); 
            //lcd.print(velValue); 
            //lcd.print("  "); 
            keyTimer[i]=0; 
          }  
          else if(keyActive[i]==0 && lastKeyActive[i]==1) { turnNoteOff(i+midiNoteStart); lastKeyActive[i]=0; keyTimer[i]=0; }  // Note off since last time  
     }
   }  
   else // mono mode
   {
     for (int i=0; i<13; i++)  // Go through the 13 notes (0..12) on the keyboard
     {
           if(keyActive[i]==1 && lastKeyActive[i]==0) // Note on since last time
          { 
            velValue=15+((keyTimer[i]-64)<<1);
            if(velValue<0) { velValue=0;} else if(velValue>127) { velValue=127; }
            lastNoteVal=i+midiNoteStart;  // store which key is last on in the loop
            lastKeyActive[i]=1; 
            //lcd.setCursor(0,0);
            //lcd.print(keyTimer[i]); 
            //lcd.print("  "); 
            //lcd.print(velValue); 
            //lcd.print("  "); 
            keyTimer[i]=0; 
          }  
          else if(keyActive[i]==0 && lastKeyActive[i]==1) { lastKeyActive[i]=0; keyTimer[i]=0; }  // Note off since last time - just store key status, no note off 
     }
     if (lastNoteVal!=0) // We have a new note
     {
        turnNoteOff(currentNoteVal);       // Turn off the last one playing
        turnNoteOn(lastNoteVal,velValue);  // Turn on the new note   
        currentNoteVal=lastNoteVal;        // Store for next time
     }
   }
}

void scanButtons() {
  digitalWrite(17,LOW);  // Turn buttons row low
  input_port=PIND;
  
  buttonActive[0]=(input_port & B00000100)>>2;  // oct down
  buttonActive[1]=(input_port & B00001000)>>3;  // oct up
  buttonActive[2]=(input_port & B00010000)>>4;  // prog down
  buttonActive[3]=(input_port & B00100000)>>5;  // prog up
  buttonActive[4]=(input_port & B01000000)>>6;  // mode
  buttonActive[5]=(input_port & B10000000)>>7;  // function

  digitalWrite(17,HIGH);  // Turn high = inactive

  for (int i=0; i<6; i++)  // check for changes
   {
      if(buttonActive[i]==0 && lastButtonActive[i]==0) // Button pressed since last time (0=active for input, 1=active for last array)
      { 
        lastButtonActive[i]=1; 
        
        if(i==0)  {
            turnNoteOff(currentNoteVal);
            globTranspose-=1;   // Go down an octave
            if(globTranspose<-3) { globTranspose=-3; }  // If we're beyond 3 octaves down then reset to 3 octaves down
            if (globTranspose==0) { octDownLED=0; octUpLED=0; octUpRate=0; octDownRate=0; }  // No lights as we're not transposed
            else if (globTranspose<0)  { octDownRate=-globTranspose; octUpRate=0; }  // Set octave down LED going at appropriate speed, turn off Up LED
            else { octUpRate=globTranspose; octDownRate=0; }  // Set octave up LED going at appropriate speed, turn off down LED
          }
        else if(i==1)  {
            turnNoteOff(currentNoteVal);
            globTranspose+=1;   // Go up an octave
            if(globTranspose>3) { globTranspose=3; }  // If we're beyond 3 octaves up then reset to 3 octaves up
            if (globTranspose==0) { octDownLED=0; octUpLED=0; octUpRate=0; octDownRate=0; }  // No lights as we're not transposed
            else if (globTranspose<0)  { octDownRate=-globTranspose; octUpRate=0; }  // Set octave down LED going at appropriate speed, turn off Up LED
            else { octUpRate=globTranspose; octDownRate=0; }  // Set octave up LED going at appropriate speed, turn off down LED
        }
        else if(i==2)  {
            progNumber--;
            if(progNumber<1) { progNumber=1; }  // If we're beyond 1 prog, set to 1
            sendProgChange();
        }
        else if(i==3)  {
            progNumber++;
            if(progNumber>127) { progNumber=127; }  // If we're beyond 127 prog, set to 127
            sendProgChange();
        }
        else if(i==4)  {
            playMode=~playMode;  // invert the playmode value
            if(playMode==0) { modeLED=0; modeRate=0; turnNoteOff(currentNoteVal); }  // turn LED off and turn off any previously sounding note
            else { modeRate=1;  }  // start LED throbbing 
            
            // TODO - change code to sustain last note on
        }
        else if(i==5)  {
           // TODO - implement function switch mode
        }

      }  
      else if(buttonActive[i]==1 && lastButtonActive[i]==1) // Button released since last time - shouldn't need to do anything but track it 
      { 
        lastButtonActive[i]=0; 
      }  
   }
}

void doPWM() {
  if (modeRate)  // If the rate is less than zero then 
  {
    analogWrite(modeLEDPin, (modeLED>>2));  // Counting will be a lot quicker than the output we want, hence the bit shift
    modeLED+=modeRate;
    if (modeLED>1000) { modeRate=-modeRate; modeLED=1000; }  // reverse the direction if fully on
    if (modeLED<0) { modeRate=-modeRate; modeLED=0; }  // reverse the direction if fully off
  }
  else { analogWrite(modeLEDPin, 0);  }  // turn LED off if there's no rate
  
  if (octDownRate)  // If the rate is less than zero then 
  {
    analogWrite(octDownLEDPin, (octDownLED>>2));  // Counting will be a lot quicker than the output we want, hence the bit shift
    octDownLED+=octDownRate;
    if (octDownLED>1000) { octDownRate=-octDownRate; octDownLED=1000; }  // reverse the direction if fully on
    if (octDownLED<0) { octDownRate=-octDownRate; octDownLED=0; }  // reverse the direction if fully off  
  }
  else { analogWrite(octDownLEDPin, 0);  }  // turn LED off if there's no rate

  if (octUpRate)  // If the rate is less than zero then 
  {
    analogWrite(octUpLEDPin, (octUpLED>>2));  // Counting will be a lot quicker than the output we want, hence the bit shift
    octUpLED+=octUpRate;
    if (octUpLED>1000) { octUpRate=-octUpRate; octUpLED=1000; }  // reverse the direction if fully on
    if (octUpLED<0) { octUpRate=-octUpRate; octUpLED=0; }  // reverse the direction if fully off  
  }
  else { analogWrite(octUpLEDPin, 0);  }  // turn LED off if there's no rate
}

void updateLCD()
{
 if(globTranspose!=lastGlobTranspose) 
 { 
   lastGlobTranspose=globTranspose; lcd.setCursor(6,1); 
   if(globTranspose==0) { lcd.print(" 0"); }
   else if(globTranspose<0) { lcd.print(globTranspose); } 
   else { lcd.print("+"); lcd.print(globTranspose); } // if neither zero or negative, must be positive
 }

 if(progNumber!=lastProgNumber) 
 { 
   lastProgNumber=progNumber; lcd.setCursor(1,1);
   if(progNumber<100) { lcd.print(" "); } // pad up spaces
   if(progNumber<10) { lcd.print(" "); } // pad up spaces
   lcd.print(progNumber);
 }

 if(playMode!=lastPlayMode) 
 { 
   lastPlayMode=playMode; lcd.setCursor(12,1);
   if(playMode==0) { lcd.print("POLY"); }
   else { lcd.print("MONO"); }
 }

  // to do - midi channel
}

void sendProgChange() {
    mySerial.print((0xC0 + midichan-1), BYTE); // C0=Program change on channel 1, plus midi channel-l
    mySerial.print(progNumber, BYTE); // New program number
    if(DebugMidi)
    {
     Serial.print((0xC0 + midichan-1),HEX); // C0=Program change on channel 1, plus midi channel-l
     Serial.print(progNumber,HEX); // New program number
     Serial.print("\n");
    }
}

void turnNoteOff(byte noteval) {
    noteOn(midichan,noteval+(globTranspose*12),0);    // Actually is a note off (velocity 0)
}

void turnNoteOn(byte noteval,byte velvalue) {
    noteOn(midichan,noteval+(globTranspose*12),velvalue);    // Note on with specified velocity
}

// Send a MIDI note-on message.  If the velocity is zero then this is interpreted as a note off
void noteOn(byte channel, byte note, byte velocity) {
  midiMsg( (0x90 + channel-1), note, velocity);
}

// Send a general MIDI message
void midiMsg(byte cmd, byte data1, byte data2) {
  mySerial.print(cmd, BYTE);
  mySerial.print(data1, BYTE);
  mySerial.print(data2, BYTE);
  if(midiBridge)
  {
    Serial.print(cmd, BYTE);
    Serial.print(data1, BYTE);
    Serial.print(data2, BYTE);
  }
  if(DebugMidi)
  {
  Serial.print(cmd,HEX);
  Serial.print(data1,HEX);
  Serial.print(data2,HEX);
  Serial.print("\n");
  }
}
